/*
 * Copyright (c) 2005, 2006, Oracle. All rights reserved.  
 */
 
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.logging.LogManager;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import oracle.jcr.OracleRepository;
import oracle.jcr.OracleRepositoryFactory;

import oracle.jcr.xdb.XDBRepositoryConfiguration;

import oracle.jdbc.pool.OracleDataSource;

/**
 * Gets the binary content of the specified resource, either copying it to
 * the local file system or sending it to stdout.
 */
public class GetContent
{
  /**
   * Entry point from command line.
   * 
   * @param args            command-line arguments
   * 
   * @throws Exception if the operation fails
   */
  public static void main(String[] args)
    throws Exception
  {
    // The XML DB Content Connector uses JDK logging.  To enable logging,
    // comment out the following line.  For further information on JDK
    // logging, refer to:
    //   http://java.sun.com/j2se/1.4.2/docs/guide/util/logging/index.html
    LogManager.getLogManager().reset();

    // Extract the command-line arguments.
    if (args == null || args.length < 4 || args.length > 5)
    {
      System.out.println("Use: java GetContent " + 
        "<databaseURL> <userName> <password> <srcPath> [ <dstPath> ]");

      System.exit(-1);
    }
    
    String databaseURL = args[0];
    String userName = args[1];
    String password = args[2];
    String srcPath = args[3];
    String dstPath = (args.length == 5) ? args[4] : null;

    // Create a JCR session.
    XDBRepositoryConfiguration repositoryConfiguration =
      new XDBRepositoryConfiguration();
    
    OracleDataSource dataSource = (OracleDataSource)repositoryConfiguration.getDataSource();
    dataSource.setURL(databaseURL);

    OracleRepository repository =
      OracleRepositoryFactory.createOracleRepository(repositoryConfiguration);

    Session session = repository.login(
      new SimpleCredentials(userName, password.toCharArray()), null);
      
    // Get the source file.
    Item item = session.getItem(srcPath);
    
    if (item instanceof Node &&
        ((Node)item).getPrimaryNodeType().getName().equals("nt:file"))
    {
      // It's an nt:file node.  So get the content.
      Property property = ((Node)item).getProperty("jcr:content/jcr:data");

      OutputStream out;
      
      if (dstPath == null)
      {
        out = System.out;
      }
      else
      {
        out = new FileOutputStream(dstPath);
      }
        
      InputStream in = property.getStream();
      
      try
      {
        byte[] buffer = new byte[32768];
        
        while (true)
        {
          int count = in.read(buffer);
          
          if (count == -1)
          {
            // EOF
            break;
          }
          
          out.write(buffer, 0, count);
        }
      }
      finally
      {
        in.close();
        
        if (dstPath != null)
        {
          out.close();
        }
      }
    }
      
    // Disconnect.
    session.logout();
    System.out.println("Disconnected");
  }
}

// EOF
